<?php

namespace App\Http\Controllers;
use App\Models\categorie;
use App\Models\produit;
use App\Models\fournisseur;
use App\Models\stock;
use App\Models\caisse;
use App\Models\taille;
use App\Models\mouvementtransfer;
use App\Models\couleur;
use App\Models\Topcat;
use App\Models\lieustock;
use App\Models\hystoriqprix;
use Illuminate\Http\Request; 
use App\Models\stock_produit;
use App\Models\produit_variable;
use Validator;

class ProduitController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */

 

public function produitparcat($lieustock_id)
{
    $stock = lieustock::findOrFail($lieustock_id);

    // Récupérer toutes les variables de produit pour ce stock
    $produitVariables = produit_variable::with(['produit', 'couleur', 'taille'])
                        ->where('lieustock_id', $lieustock_id)
                        ->get();

    // Récupérer les produits uniques
    $produits = $produitVariables->pluck('produit')->unique('id')->values();

    // Charger les variables pour chaque produit
    foreach ($produits as $produit) {
        $produit->load(['variables' => function($q) use ($lieustock_id) {
            $q->where('lieustock_id', $lieustock_id)->with(['couleur', 'taille']);
        }]);
    }

    return view('admin.produit.produitparcat', compact('produits', 'stock'));
}


    // PAGE 2 : Voir les couleurs d'un produit dans un stock
    public function couleur($produit_id, $lieustock_id)
    {
        $produit = produit::findOrFail($produit_id);

        $couleurs_ids = produit_variable::where('produit_id', $produit_id)
                                        ->where('lieustock_id', $lieustock_id)
                                        ->distinct()
                                        ->pluck('couleur_id');

        $couleurs = couleur::whereIn('id', $couleurs_ids)->get();
        $stock = lieustock::findOrFail($lieustock_id);

        return view('admin.produit.couleur', compact('produit','couleurs','stock'));
    }

    // PAGE 3 : Voir les tailles d'une couleur pour un produit dans un stock
    public function taille($produit_id, $couleur_id, $lieustock_id)
    {
        $produit = produit::findOrFail($produit_id);
        $couleur = couleur::findOrFail($couleur_id);
        $stock = lieustock::findOrFail($lieustock_id);

        $variantes = produit_variable::where('produit_id', $produit_id)
                        ->where('couleur_id', $couleur_id)
                        ->where('lieustock_id', $lieustock_id)
                        ->with('taille','lieustock')
                        ->get();

        $lieux = lieustock::all();

        return view('admin.produit.taille', compact('produit','couleur','variantes','lieux','stock'));
    }

    // Transférer qty vers un autre stock
public function transfer(Request $request, $id)
{
    // Récupérer le produit_variable avec sa relation lieustock
    $produit_variable = produit_variable::with('lieustock')->findOrFail($id);

    $request->validate([
        'qty' => 'required|integer|min:1',
        'lieustock_id' => 'required|exists:lieustocks,id',
    ]);

    $qtyToTransfer = (int) $request->qty;
    $currentQty = (int) $produit_variable->qty;

    $stockOrigine = $produit_variable->lieustock->lieu ?? 'Inconnu';
    $stockDestination = lieustock::find($request->lieustock_id)->lieu ?? 'Inconnu';

    if ($currentQty < $qtyToTransfer) {
        return back()->with('error', "Quantité insuffisante ! Stock actuel ($stockOrigine): $currentQty, demandé: $qtyToTransfer vers stock '$stockDestination'");
    }

    // Décrémenter stock origine
    $produit_variable->qty -= $qtyToTransfer;
    $produit_variable->save();

    // Ajouter au stock destination (ou créer s'il n'existe pas)
    $target = produit_variable::firstOrCreate([
        'produit_id' => $produit_variable->produit_id,
        'couleur_id' => $produit_variable->couleur_id,
        'taille_id' => $produit_variable->taille_id,
        'lieustock_id' => $request->lieustock_id,
    ], ['qty' => 0]);

    $target->qty += $qtyToTransfer;
    $target->save();
        // Enregistrer le mouvement
    mouvementtransfer::create([
        'produit_id' => $produit_variable->produit_id,
        'couleur_id' => $produit_variable->couleur_id,
        'taille_id' => $produit_variable->taille_id,
        'stock_origine_id' => $produit_variable->lieustock_id,
        'stock_destination_id' => $request->lieustock_id,
        'qty' => $qtyToTransfer,
    ]);

    return back()->with('success', "Quantité transférée avec succès ! Stock origine: '$stockOrigine', Stock destination: '$stockDestination'");
}



public function storeCouleurTaille(Request $request, $produit_id)
{
    $request->validate([
        'couleur_id' => 'required|exists:couleurs,id',
        'taille_id' => 'required|exists:tailles,id',
        'qty' => 'required|integer|min:0',
        'lieustock_id' => 'required|exists:lieustocks,id',
    ]);

    // Vérifier si la combinaison existe déjà pour ce produit dans CE stock
    $variableExist = produit_variable::where('produit_id', $produit_id)
        ->where('couleur_id', $request->couleur_id)
        ->where('taille_id', $request->taille_id)
        ->where('lieustock_id', $request->lieustock_id)
        ->exists();

    if ($variableExist) {
        return back()->with('error', '⚠️ Cette couleur et taille existe déjà pour ce produit dans ce stock !');
    }

    // Créer la nouvelle ligne
    produit_variable::create([
        'produit_id' => $produit_id,
        'couleur_id' => $request->couleur_id,
        'taille_id' => $request->taille_id,
        'lieustock_id' => $request->lieustock_id,
        'qty' => $request->qty,
    ]);

    return back()->with('success', '✅ Couleur/Taille ajoutée avec succès !');
}



    
     // Affiche le formulaire
    public function create()
    {
        return view('admin.produit.createproduit');
    }

    // Stocke le produit
public function store(Request $request)
{
    $request->validate([
        'nom' => 'required|string|max:255',
        'img' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048',
        'prixun' => 'nullable|numeric',
        'prixvents' => 'nullable|numeric',
        'topcat_id' => 'required|exists:topcats,id',
    ]);

    // Création du produit
    $produit = Produit::create([
        'user_id' => auth()->id(),
        'topcat_id' => $request->topcat_id,
        'nom' => $request->nom,
        'prixun' => $request->prixun,
        'prixvents' => $request->prixvents,
        'img' => $request->hasFile('img') 
                 ? 'produits/'. $request->file('img')->store('produits', 'public') 
                 : null,
    ]);

    // Ajouter le produit dans tous les lieustocks
    $lieustocks = lieustock::all();
    foreach ($lieustocks as $ls) {
        produit_variable::create([
            'produit_id' => $produit->id,
            'lieustock_id' => $ls->id,
            'taille_id' => null,
            'couleur_id' => null,
            'qty' => null,
        ]);
    }

    return redirect()->route('admin.produit.create')->with('success', 'Produit ajouté dans tous les lieux de stock!');
}


public function filter(Request $request)
{
    $lieux = lieustock::all();
    $categories = Topcat::all();

    $query = produit_variable::with(['produit.topcat', 'couleur', 'taille', 'lieustock']);

    if ($request->filled('lieustock_id')) {
        $query->where('lieustock_id', $request->lieustock_id);
    }

    if ($request->filled('categorie_id')) {
        $query->whereHas('produit', function($q) use ($request) {
            $q->where('topcat_id', $request->categorie_id);
        });
    }

    $produits = $query->get();

    return view('admin.produit.filtre', compact('produits','lieux','categories'));
}


}
     
     




     
     



